<?php
/**
 * @file
 * Hooks for Cforge distribution profile.
 *
 * @todo ensure all lists show only active users with role RID_TRADER
 */

use Drupal\node\Entity\Node;
use Drupal\taxonomy\Entity\Vocabulary;
use Drupal\user\UserInterface;
use Drupal\user\Entity\User;
use Drupal\Core\Render\Element;
use Drupal\Core\Extension\Extension;
use Drupal\Core\Entity\Entity\EntityFormDisplay;
use Drupal\Core\Url;
use Drupal\Core\Access\AccessResult;

// Roles.
const RID_TRADER = 'trader';
const RID_COMMITTEE = 'committee';
const CFORGE_FLAVOURS = [
  'hamlets' => 'Hamlets',
  'ces' => 'CES',
  'localexchange' => 'Local Exchange',
];


/**
 * Implements hook_form_alter().
 */
function cforge_form_alter(&$form, $form_state, $form_id) {
  switch ($form_id) {
    case 'user_login_form':
    case 'user_login_block':
      // Redirect the user to given destination or else the member front page
      $request = \Drupal::request();
      if (!$request->request->has('destination')) {
        $path = \Drupal::config('cforge.settings')->get('member_frontpage');
        $request->query->set('destination', $path);
      }
      break;
    case 'user_pass':
      // $form_state->setRedirect doesn't work while the form is being built.
      \Drupal::request()->query->set('destination', '/user/login');
      break;
    case 'user_form':
    case 'user_register_form':
      // Css lays out the given_name and family_name fields better
      $form['#attached']['library'][] = 'cforge/cforge_css';
      break;
    case  'user_profile_form';
      // Unhide this which was hidden in Drupal\address\Plugin\Field\FieldWidget\AddressDefaultWidget::addressElements()
      unset($form['address']['widget'][0]['address_line2']['#title_display']);
      $form['#attached']['library'][] = 'cforge/cforge_css';
  }
}

/**
 * Implements hook_block_view_BASE_BLOCK_ID_alter().
 *
 * Change the title of the user block.
 */
function cforge_block_view_system_menu_block_alter(&$build, $block) {
  if ($block->getPluginId() == 'system_menu_block:account') {
    $build['#configuration']['label'] = \Drupal::currentUser()->getDisplayName();
  }
}

/**
 * Implements hook_cron().
 */
function cforge_cron() {
  // Delete old sessions to keep the sessions table small.
  db_delete('sessions')
    ->condition('uid', 0)
    ->condition('timestamp', strtotime('-1 day'), '<')
    ->execute();
  db_delete('sessions')
    ->condition('timestamp', strtotime('-1 month'), '<')
    ->execute();
  // Keep the watchdog table small too.
  if (\Drupal::moduleHandler()->moduleExists('watchdog')) {
    db_delete('watchdog')
      ->condition('timestamp', strtotime('-1 week'), '<')
      ->execute();
  }

  $conf = \Drupal::configFactory()->getEditable('cforge.settings');
  if ($conf->get('demo_mode')) {
    // Restore from backup, if we are in demo mode on the first cron of every day
    if (date('z', $conf->get('last_cron')) <> date('z', REQUEST_TIME)) {
      $info = Drupal\Core\Database\Database::getConnectionInfo();
      $db  = reset($info);
      $command = 'mysql -u '.$db['username'];
      if ($db['password']) {
        $command .= ' -p '.$db['password'];
      }
      $command .= ' '.$db['database'] .' < '.\Drupal::service('file_system')->realpath('private://dump.sql');
      exec($command);
      // This restoration hasn't deleted any files which were uploaded. We might use bash find for that.
    }
  }
  $conf->set('last_cron', REQUEST_TIME)->save();
}

/**
 * Implements hook_element_info_alter().
 *
 * This is all the code from the disablepwstrength module.
 */
function cforge_element_info_alter(&$types) {
  $processes = &$types['password_confirm']['#process'];
  if (isset($processes)) {
    $position = array_search('user_form_process_password_confirm', $processes);
    if ($position !== FALSE) {
      unset($processes[$position]);
    }
  }
}

/**
 * Implements hook_entity_info_alter().
 *
 * Add new Form handlers for the user profile edit forms.
 */
function cforge_entity_type_alter(array &$entity_types) {
  if (isset($entity_types['user'])) {
    $entity_types['user']->setFormClass('profile', "Drupal\user\ProfileForm");
    $entity_types['user']->setFormClass('admin', "Drupal\user\ProfileForm");
    $entity_types['user']->setAccessClass('Drupal\cforge\NeighbourhoodUserAccess');
  }
  if (isset($entity_types['smallad'])) {
    $entity_types['smallad']->setAccessClass('Drupal\cforge\NeighbourhoodSmalladAccess');
  }
  if (isset($entity_types['smallad'])) {
    $entity_types['mcapi_wallet']->setAccessClass('Drupal\cforge\NeighbourhoodWalletAccess');
  }
  // @todo apart from these three, should super neighbours be able to use wallets?

}

/**
 * Implements hook_form_FORM_ID_alter().
 */
function cforge_form_user_form_alter(&$form, $form_state) {
  if (isset($form['account'])) {
    // @see cforge_entity_base_field_info_alter();
    unset($form['account']['current_pass']);
    $form['account']['status']['#options'][0] = t('User has not been approved or is no longer a member');
    $form['account']['status']['#options'][1] = t('User can log in normally');
    $form['account']['roles']['#options'][RID_COMMITTEE] .= ' - ' . (string) t('Can manage members, articles, pages images, & documents.');
    $form['account']['roles']['#options'][RID_TRADER] .= ' - ' . t('Can use the main currency.');
  }
  // Remove the mutiple field theme from the phone numbers.
  $form['phones']['widget'][0]['_weight']['#access'] = FALSE;
  $form['phones']['widget'][1]['_weight']['#access'] = FALSE;
  unset($form['phones']['widget']['#theme']);

  if (isset($form['picture'])) {
    $helpful = t('No spaces in filenames!');
    $form['picture']['picture_upload']['#title'] .= ' (' . $helpful . ')';
  }
  //hide the 'present' checkbox for user 1
  if ($form_state->getFormObject()->getEntity()->id() == 1 and isset($form['present'])) {
    $form['present']['#access'] = FALSE;
  }

}

/**
 * Implements hook field_widget_WIDGET_NAME_form_alter().
 *
 * Tailor the display of the phone multiple cardinality field.
 */
function cforge_field_widget_telephone_default_form_alter(&$element, $form_state, $context) {
  static $i = 0;
  $element['value']['#field_prefix']  = ($i % 2 == 1) ? t('Other phone:') : t('Mobile phone:');
  $i++;
}

/**
 * Implements hook field_widget_WIDGET_NAME_form_alter().
 *
 * Reformat the address widget assuming there is no choice of country and
 * neighbourhood is a dropdown. Restrict the available countries to those in
 * config.
 */
function cforge_field_widget_address_default_form_alter(&$element, $form_state, $context) {
  $element['address']['#type'] = 'cforge_address';
}

/**
 * Implements hook_form_FORM_ID_alter().
 *
 * Tailor the account settings form.
 */
function cforge_form_user_admin_settings_alter(&$form) {
  $reg_fieldset = &$form['registration_cancellation'];
  unset(
    $reg_fieldset['#description'],
    $reg_fieldset['user_register']['#options'][UserInterface::REGISTER_VISITORS]
  );
  $reg_fieldset['#weight'] = -5;
  $reg_fieldset['user_register']['#title'] = t('How to add new members');
  $reg_fieldset['user_register']['#options'][UserInterface::REGISTER_ADMINISTRATORS_ONLY] = t('Only committee can create accounts');
  $reg_fieldset['user_register']['#options'][UserInterface::REGISTER_VISITORS_ADMINISTRATIVE_APPROVAL] = t('Anyone can create an account, then a committee member must enable it');
  $reg_fieldset['user_cancel_method']['#access'] = FALSE;
  $reg_fieldset['user_email_verification']['#access'] = FALSE;
  $form['contact']['#access'] = FALSE;
  $form['admin_role']['#access'] = FALSE;
}

/**
 * Implements hook_form_FORM_ID_alter().
 *
 * Other node types have no menus at all.
 */
function cforge_form_node_page_form_alter(&$form, &$form_state) {
  $home = \Drupal::config('system.site')->get('page.front');
  // Prevent changing the alias of the home page, and remove the delete button.
  if ($form['path']['widget'][0]['alias']['#default_value'] == $home) {
    $form['path']['warning'] = [
      '#markup' => t(
        "Warning: '@name' is your site's front page.",
        ['@name' => $home]
      ),
    ];
  }
}

/**
 * Implements hook_form_node_form_alter().
 *
 * Make the descriptions of the aliases more friendly.
 */
function cforge_form_node_form_alter(&$form, $form_state) {
  $node = $form_state->getFormObject()->getEntity();
  if ($node->bundle() == 'page') {
    $form['public'] = [
      '#title' => t('Visible to the public'),
      '#type' => 'checkbox',
      '#default_value' => cforge_node_get_anonaccess($node->id()),
    ];
    foreach (Element::children($form['actions']) as $name) {
      $actions[$name]['#submit'][] = 'cforge_node_submit';
    }
  }
  // Rename the url alias field and explain it better.
  $form['path']['widget'][0]['alias']['#title'] = t('Friendly address');
  $form['path']['widget'][0]['alias']['#description'] = t(
    "Optionally give your page a more meaningful address. E.g. 'articles-of-association'"
  );
}

/**
 * Implements hook_menu_links_discovered_alter().
 *
 * Remove unwanted menu items put in the main menu by core modules.
 */
function cforge_menu_links_discovered_alter(&$links) {
  unset(
    // Poor default in the node module.
    $links['node.add_page'],
    // Was disabled anyway.
    $links['filter.tips_all'],
    // Was disabled anyway.
    $links['search.view'],
    // @see Drupal\cforge\Routing\RouteSubscriber
    // $links['masquerade.unmasquerade'],//access bug in masquerade module beta
    // created by admin_toolbar_tools module
    $links['entity.menu.add_form'],
    $links['views_view:views.smallads_directory.page_list']
  );
  $links['contact.site_page']['enabled'] = TRUE;
  // Having removed node.add_page its is helpful to rename the links under it.
  if  (\Drupal::entityTypeManager()->getDefinition('node_type', FALSE)) {// may not be installed yet.
    foreach (\Drupal::entityTypeManager()->getStorage('node_type')->loadMultiple() as $type) {
      $type_id = $type->id();
      if (isset($links['node.add.' . $type_id])) {
        $links['node.add.' . $type_id]['title'] = t(
          'Add @type',
          ['@type' => $links['node.add.' . $type_id]['title']]
        );
      }
    }
  }
  // This works even before taxonomy is installed.
  foreach (\Drupal::configFactory()->listAll('taxonomy.vocabulary') as $id) {
    $vocab = Vocabulary::load(substr($id, 20));
    $vid = $vocab->id();
    $links['entity.taxonomy_vocabulary.overview_form.cforge.' . $vid] = [
      'title' => t('Manage @name', ['@name' => $vid]),
      'description' => $vocab->getDescription(),
      'route_name' => 'entity.taxonomy_vocabulary.overview_form',
      'route_parameters' => ['taxonomy_vocabulary' => $vid],
      'menu_name' => 'tools',
      'weight' => 5
    ];
    // The admin_toolbar_tools module puts its own links...
    $key = 'entity.taxonomy_term.add_form.' . $vid;
    if (isset($links[$key])) {
      unset($links[$key]);
    }
  }
  if (isset($links['user.logout'])) {
    // This overrides admin_toolbar_tools_menu_links_discovered_alter()
    unset($links['user.logout']['parent']);
    $links['user.logout']['menu_name'] = 'account';
  }
  if (isset($links['mcapi.mass'])) {
    $links['mcapi.mass']['weight'] = 1;
  }
  if (isset($links['views_view:views.smallads_directory.collection'])) {
    $links['views_view:views.smallads_directory.collection']['menu_name'] = 'tools';
  }
}

/**
 * Implements hook_menu_local_actions_alter().
 *
 * @note the the menu links, actions and task that refer to it are also removed.
 */
function cforge_menu_local_actions_alter(&$items) {
  // @see Drupal\cforge\Routing\RouteSubscriber
  unset($items['entity.menu.add_form']);
}

/**
 * Implements hook_local_tasks_alter().
 */
function cforge_local_tasks_alter(&$items) {
  unset($items['smallads.admin']);//this refers to a view that this distro doesn't use.
}

/**
 * Implements hook_mail().
 */
function cforge_mail($key, &$message, $params) {
  if ($key == 'all') {
    $recipient = User::loadByProperties(['mail' => $message['to']]);
    // Tokens allowed were.
    $message['subject'] = $params['subject'];
    $message['body'][] = \Drupal::token()->replace($params['body'], ['user' => $recipient]);
  }
}

/**
 * Implements hook_mail_alter().
 *
 * Add the mail footer.
 */
function cforge_mail_alter(&$message) {
  if (!is_array($message['body'])) {
    // We assume it is a string.
    drupal_set_message('problem constructing body of mail '.$message['key'], 'warning');
    $message['body'] = (array)$message['body'];
  }
  if (\Drupal::currentUser()->id()) {
    $message['body'][] = \Drupal::token()->replace(
      \Drupal::config('cforge.settings')->get('mail_footer'),
      $message['params'],
      [
        'language' => $message['langcode'],
        'sanitize' => FALSE,
      ]
    );
  }
  if (\Drupal::config('cforge.settings')->get('demo_mode')) {
    // Block the mail and print to screen.
    $message['send'] = FALSE;
    $subject = (string)t('Email intercepted: @subject', ['@subject' => $message['subject']]);
    drupal_set_message($subject ."\n\n". implode("\n\n", $message['body']));
  }
}

/**
 * Implements hook_form_system_site_information_settings_alter().
 */
function cforge_form_system_site_information_settings_alter(&$form, $form_state) {
  $form['front_page']['member_frontpage'] = [
    '#title' => t("Member's front page"),
    '#description' => t("Users will be redirected to this page when they log in."),
    '#default_value' => \Drupal::config('cforge.settings')->get('member_frontpage'),
    '#element_validate' => ['cforge_member_frontpage_validate'],
  ] + $form['front_page']['site_frontpage'];
  $form['#submit'][] = 'cforge_submit_system_site_information_settings';
}

/**
 * Element validation callback.
 */
function cforge_member_frontpage_validate(&$element, $form_state) {
  if ($element['#value'] && $element['#value'][0] !== '/') {
    $form_state->setError($element, t("The path '%path' has to start with a slash.", ['%path' => $element['#value']]));
  }
}

/**
 * Form submission callback.
 */
function cforge_submit_system_site_information_settings($form, $form_state) {
  \Drupal::configFactory()->getEditable('cforge.settings')
    ->set('member_frontpage', $form_state->getValue('member_frontpage'))
    ->save();
}

/**
 * Implements hook_locale_translation_projects().
 *
 * To clear the locale_file table and on devel/php do
 *
 *  \Drupal::moduleHandler()->loadInclude('locale', 'fetch.inc');
 *  \Drupal::moduleHandler()->loadInclude('locale', 'bulk.inc');
 *  locale_translation_clear_status();
 *  $batch = locale_translation_batch_update_build(array(), $langcodes, array());
 *  batch_set($batch);
 *
 * @see locale_translation_check_projects_local()
 */
function cforge_locale_translation_projects_alter(&$projects) {
  foreach (['cforge', 'smallads', 'mutual_credit', 'sky_seldulac'] as $module) {
    $projects[$module]['server_pattern'] = 'http://translations.communityforge.net/8.x/%project/%version.%language.po';
  }
}

/**
 * Implements hook_form_system_modules_alter().
 *
 * Disable the cf_ladmin checkbox, if present.
 * Add Readme links to the modules page, where readme files exist.
 * Prevent the multiple exchanges module being installed.
 */
function cforge_form_system_modules_alter(&$form, &$form_state) {
  // Protect the cforge hosted module (only applies on cforge server)
  if (array_key_exists('cf_ladmin', $form['modules']['Other'])) {
    if (\Drupal::currentUser()->id() != 1) {
      $form['modules']['Other']['cf_ladmin']['enable']['#disabled'] = TRUE;
      $form['modules']['Other']['cf_ladmin']['description']['#markup'] .= strtoupper(t('Only user 1 can disable this module'));
    }
  }
  unset($form['modules']['Other']['cforge_import']);
  // Link to the readme file. (cool idea taken from the advanced help module).
  foreach (Element::children($form['modules']) as $group) {
    foreach (Element::children($form['modules'][$group]) as $module) {
      $path = DRUPAL_ROOT . '/' . drupal_get_path('module', $module);
      if (file_exists($path . '/README.txt')) {
        $form['modules'][$group][$module]['links']['help'] = [
          '#type' => 'link',
          '#title' => t('README.txt'),
          '#url' => Url::fromRoute('cforge.readme', ['module' => $module]),
          '#attributes' => [
            'class' => ['module-link'],
          ],
        ];
      }
    }
  }
  // Mcapi_exchanges module is likely to be incompatible with Cforge profile.
  unset($form['modules'][(string) t('Community Accounting')]['mcapi_exchanges']);
  // Impossible to change flavour so don't even show the flavours as modules.
  foreach (CFORGE_FLAVOURS as $mod => $name) {
    unset($form['modules']['Other'][$mod]);
  }
}

/**
 * Implements hook_system_info_alter().
 *
 * Move things around on the modules page.
 */
function cforge_system_info_alter(&$info, Extension $extension, $type) {
  // Make some contrib modules into cforge features!
  switch ($extension->getName()) {
    case 'community_tasks':
      $info['package'] = (string) t('Cforge features');
  }
}

/**
 * Implements hook_form_alter().
 *
 * Clears the css cache after a modification.
 *
 * @todo test when css_injector is available
 */
function cforge_form_css_injector_edit_alter(&$form, &$form_state) {
  $form['buttons']['save']['#submit'][] = 'drupal_clear_css_cache';
  $form['buttons']['save_and_continue']['#submit'][] = 'drupal_clear_css_cache';
}

/**
 * Utility function to make batch mail a little easier for cforge dependencies.
 *
 * @param string $module
 *   The name of the modue which is responsible for the mail.
 * @param string $key
 *   The key of that mail.
 * @param array $uids
 *   The Uids of the recipients.
 * @param array $params
 *   The mail params.
 * @param \Drupal\Core\Url|string $redirect
 *   Where to redirect after tha batch is run.
 * @param string $finished
 *   The path to go to at the end of the batch.
 *
 * @todo test batch mail in d8.
 */
function cforge_batch_mail($module, $key, array $uids, $params, $redirect, $finished = '') {
  if (empty($uids)) {
    return;
  }
  $batch = [
    'title' => t('Sending @count mails', ['@count' => count($uids)]),
    // @see https://api.drupal.org/api/drupal/core!includes!form.inc/function/batch_process/8
    'batch_redirect' => $redirect,
  ];
  if ($finished) {
    $batch['finished'] = $finished;
  }
  foreach ($uids as $uid) {
    $batch['operations'][] = [
      'cforge_batch_mail_send',
      [$module, $key, $uid, $params],
    ];
  }
  batch_set($batch);
}

/**
 * Batch callback.
 */
function cforge_batch_mail_send($module, $key, $uid, $params, $progress) {
  $account = user_load($uid);
  $params['user'] = $account;
  \Drupal::service('plugin.manager.mail')->mail(
    $module,
    $key,
    $account->getEmail(),
    $account->getPreferredLangcode(),
    $params
  );
}

/**
 * Implements hook_node_access().
 */
function cforge_node_access($node, $op, $account) {
  $result = AccessResult::neutral();
  if ($node->bundle() == 'neighbourhood') {
    // Prevent deletion or changing name of used neighbourhoods.
    if ($op == 'update' || $op == 'delete') {
      // Only allow update and delete of unused neighbourhoods
      $used = \Drupal::entityQuery('user')
        ->condition('address.dependent_locality', $form_state->getFormObject()->getEntity()->label())
        ->count()
        ->execute();
      if ($used) {
        $result = AccessResult::forbidden();
      }
    }
  }
  elseif ($op == 'view') {
    // Handle the public-only nodes.
    $public = cforge_node_get_anonaccess($node->id());
    if ($node->bundle() == 'page') {
      $result = ($public == $account->isAnonymous()) ?
        AccessResult::allowed()->cachePerPermissions() :
        AccessResult::forbidden();
    }
    else {
      $result = $public || $account->isAuthenticated() ?
        AccessResult::allowed()->cachePerPermissions() :
        AccessResult::forbidden();
    }
  }
  return $result;
}

/**
 * Theme preprocessor for smallads.
 *
 * Add the pay link(s).
 */
function cforge_preprocess_smallad(&$vars) {
  if (isset($vars['elements']['pay_links'])) {
    $vars['pay_links'] = render($vars['elements']['pay_links']);
  }
}

/**
 * Implements hook_form_taxonomy_term_categories_form_alter().
 *
 * Force the term description field to be plain text. There doesn't seem to be a
 * way of doing this by altering the baseFieldDefinition.
 */
function cforge_form_taxonomy_term_categories_form_alter(&$form, $form_state) {
  $form['description']['widget'][0]['format']['#allowed_formats'] = ['plain_text'];
}

/**
 * Form submission handler.
 */
function cforge_node_submit($form, $form_state) {
  cforge_node_set_anonaccess($form_state->getFormObject()->getEntity()->id(), $form_state->getValue('publicevent'));
}

/**
 * Implements hook_node_delete().
 */
function cforge_node_delete($node) {
  \Drupal::keyValue('anonaccess')->delete($node->id);
}

/**
 * Determine if a node is intended for anon.
 *
 * @param int $nid
 */
function cforge_node_get_anonaccess($nid) {
  return \Drupal::keyValue('anonaccess')->get($nid);
}

/**
 * Form submit callback.
 *
 * Set the anonaccess flag for a node.
 */
function cforge_node_set_anonaccess($nid, $value) {
  \Drupal::keyValue('anonaccess')
    ->set($nid, $value);
}

/**
 * Make a list of of all the neighbourhood entities.
 *
 * @return Neighbourhood[]
 *   Both keys and values are the entity label, order alphabetically.
 *
 * @todo can we use Mcapi::entityLabelList($entity_type_id);
 */
function cforge_list_neighbourhoods() {
  $options = [];
  $nids = \Drupal::entityQuery('node')
    ->condition('type', 'neighbourhood')
    ->execute();
  foreach (Node::LoadMultiple($nids) as $node) {
    $options[] = $node->label();
  }
  sort($options);
  return array_combine($options, $options);
}

/**
 * Check if the passed user lives in the same neighbourhood as the current user.
 *
 * @param \Drupal\user\UserInterface $user
 *   Fully loaded user entity.
 *
 * @result boolean
 *   True if the passed user is in the same neighbourhood as the currentUser.
 */
function cforge_same_neighbourhood(UserInterface $user) {
  // There has to be a more elegant way of addressing subfields.
  $currentuser = User::load(\Drupal::currentuser()->id());
  return $currentuser->address->getValue()[0]['dependent_locality']
    === $user->address->getValue()[0]['dependent_locality'];
}

/**
 * Determine whether the referral system is active according to cforge settings.
 *
 * @return bool
 *   TRUE if the referral fee is not 0 and there is a source wallet.
 */
function cforge_referral() {
  $conf = \Drupal::Config('cforge.settings');
  return !empty_worths_config($conf->get('referral_fee')) && $conf->get('referral_src');
}

/**
 * Implements hook_ENTITY_TYPE_insert().
 *
 * Modify default content view to remove unwanted bulk actions. Unfortunately I
 * can't see how to do this while installing. since the view hasn't been created
 * by the time of cforge_install
 */
function cforge_view_insert($view) {
  if ($view->id() == 'content') {
    $display = &$view->getDisplay('default');
    $display['display_options']['fields']['node_bulk_form']['include_exclude'] = 'include';
    $display['display_options']['fields']['node_bulk_form']['selected_actions'] = [
      'node_delete_action',
      'node_publish_action',
      'node_unpublish_action',
    ];
    $view->save();
  }
}

/**
 * Implements hook_user_presave().
 */
function cforge_user_presave($account) {
  // If the user is being created, add the trader role.
  if ($account->isNew()) {
    $account->addRole(RID_TRADER);
  }
  if ($account->id() == 1) {
    // Hide user 1 from offers/want listings
    //TODO should hide this field on the user 1 form
    $account->present->value = 0;
  }
}

/**
 * Implements hook_entity_base_field_info_alter().
 *
 * Remove the protected fields from the user and hence the need to confirm with
 * a password on the user edit form.
 */
function cforge_entity_base_field_info_alter(&$fields, $entity_type) {
  // Alter the mymodule_text field to use a custom class.
  if ($entity_type->id() == 'user') {
    foreach (['mail', 'pass'] as $fieldname) {
      $constraints = $fields[$fieldname]->getConstraints();
      unset($constraints['ProtectedUserField']);
      $fields[$fieldname]->setConstraints($constraints);
    }
  }
}


/**
 * Save a mysql dump in a constant place.
 *
 * @note there's no UI to trigger this.
 */
function cforge_backup() {
  $info = Drupal\Core\Database\Database::getConnectionInfo();
  $db  = reset($info);
  //get the files directory.
  $command = 'mysqldump --add-drop-database -u '.$db['username'].' -p '.$db['password'].' '.$db['database'];
  exec($command, $output);
  file_unmanaged_save_data(implode("\n", $output), 'private://dump.sql', FILE_EXISTS_REPLACE);
}

/**
 * Implements hook_migrate_prepare_row().
 */
function cforge_migrate_prepare_row($row, $source, $migration) {
  module_load_include('migrate.inc', 'cforge');
  return _cforge_migrate_prepare_row($row, $source, $migration);
}

 /**
  * Implements hook_migration_plugins_alter().
  */
function cforge_migration_plugins_alter(array &$definitions)  {
  module_load_include('migrate.inc', 'cforge');
  _cforge_migration_plugins_alter($definitions);
}

 /**
  * Implements hook_migrate_field_info_alter().
  */
function cforge_migrate_field_info_alter(array &$fields) {
  $fields['list']['type_map']['list_float'] = 'list_float';
}

/**
 * Implements hook_modules_installed().
 */
function cforge_modules_installed($modules) {
  if (in_array('personal_digest', $modules)) {
    if ($conf = EntityFormDisplay::load('user.user.default')) {
      // Goes to the bottom.
      $conf->setComponent('personal_digest')->save();
    }
    if ($conf = EntityFormDisplay::load('user.user.profile')) {
      $conf->removeComponent('personal_digest')->save();
    }
  }
  if (in_array('community_tasks', $modules)) {
    Role::load('trader')
      ->grantPermission('view tasks')
      ->grantPermission('commit to tasks')
      ->save();
    Role::load('committee')
      ->grantPermission('edit tasks')
      ->save();
      // Alter the default view from Community Tasks module
    $conf = \Drupal::ConfigFactory()->getEditable('views.view.current_community_tasks');
    $conf->set('display.default.display_options.access.type', 'role')
    ->set('display.default.display_options.access.options.role', ['trader' => 'trader'])
    ->save();
  }
}