<?php

namespace Drupal\cforge_import;

// This is for the validation of large sets.
ini_set('max_execution_time', 200);

use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Form builder for selecting andn importing from a csv.
 */
class ImportUrlForm extends FormBase {

  private $csvImporter;
  private $plugin;

  /**
   * Constructor.
   */
  public function __construct($csv_importer) {
    $this->csvImporter = $csv_importer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('cforge.csv_importer')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'cforge_import_csv_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['format'] = [
      '#title' => $this->t('Input format'),
      '#type' => 'radios',
      '#options' => [
        'file' => $this->t('Upload file'),
        'paste' => $this->t('Paste csv text'),
        'url' => $this->t('Absolute url'),
      ],
      '#default_value' => isset($_SESSION['import']) ? $_SESSION['import']['format'] : '',
    ];
    $form['import_type'] = [
      '#title' => $this->t('Type'),
      '#descriptions' => $this->t('Importable types'),
      '#type' => 'radios',
      '#weight' => 1,
    ];

    $options = [];
    foreach ($this->csvImporter->getDefinitions() as $id => $def) {
      $form['import_type']['#options'][$id] = $def['label'];
      $this->setPlugin($id);
      $col_names = array_keys($this->plugin->columns());
      $description = $this->t(
        'Csv columns: @columns',
        ['@columns' => implode(', ', $col_names)]
      );
      $class = get_class($this->plugin);
      if ($class::ready()) {
        $form[$id . '_info'] = [
          '#type' => 'item',
          '#markup' => $this->t('Field descriptions') . '<br />',
          '#states' => [
            'visible' => [
              ':input[name="import_type"]' => ['value' => "$id"],
            ],
          ],
          '#weight' => 4,
        ];
        foreach ($this->plugin->columns() as $field => $info) {
          $form[$id . '_info']['#markup'] .= '<strong>' . $field . '</strong>' . ': ' . $info . '<br />';
        }
        $form[$id . '_file'] = [
          '#type' => 'file',
          '#title' => $def['label'],
          '#weight' => 5,
          '#states' => [
            'visible' => [
              ':input[name="import_type"]' => ['value' => "$id"],
              ':input[name="format"]' => ['value' => "file"],
            ],
          ],
        ];
        $form[$id . '_paste'] = [
          '#title' => $this->t('Paste your csv file here'),
          '#type' => 'textarea',
          '#placeholder' => implode(', ', $col_names),
          '#weight' => 6,
          '#states' => [
            'visible' => [
              ':input[name="import_type"]' => ['value' => "$id"],
              ':input[name="format"]' => ['value' => "paste"],
            ],
          ],
        ];
        $form[$id . '_url'] = [
          '#title' => $this->t('Paste your csv file here'),
          '#placeholder' => 'http://',
          '#type' => 'textfield',
          '#weight' => 6,
          '#states' => [
            'visible' => [
              ':input[name="import_type"]' => ['value' => "$id"],
              ':input[name="format"]' => ['value' => "url"],
            ],
          ],
        ];
      }
      else {
        $form['import_type'][$id]['#disabled'] = TRUE;
      }
    }
    $form['delete'] = array(
      '#title' => t('Delete existing'),
      '#type' => 'checkbox',
      '#default_value' => TRUE,
      '#weight' => 9,
    );
    $form['test'] = array(
      '#title' => t('Creates and validate entities but do NOT save'),
      '#type' => 'checkbox',
      '#default_value' => TRUE,
      '#weight' => 9,
    );
    $form['submit'] = array(
      '#type' => 'submit',
      '#value' => 'import',
      '#weight' => 10,
    );
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    if ($form_state->getValue('format') == 'file') {
      $element_name = $form_state->getValue('import_type') . '_file';
      // Check for a new uploaded file.
      $file = file_save_upload($element_name, ['file_validate_extensions' => ['csv']], FALSE, 0);
      if (isset($file)) {
        // File upload was attempted.
        if ($file) {
          // Put the temporary file in form_values so we can save it on submit.
          $form_state->setValue($element_name, $file);
        }
        else {
          // File upload failed.
          $form_state->setErrorByName($element_name, $this->t('The csv could not be uploaded.'));
        }
      }
    }
    $type = $form_state->getValue('import_type');
    $this->setPlugin($type);
    if ($form_state->getValue('format') == 'file') {
      $file_entity  = $form_state->getValue($type . '_file');
      $data = file_get_contents($file_entity->getFileUri());
    }
    elseif ($form_state->getValue('format') == 'url') {
      $url = $form_state->getValue($type . '_url');
      $data = \Drupal::httpClient()->get($url)->getBody()->getContents();
    }
    else {
      $data = $form_state->getValue($type . '_paste');
    }
    $form_state->set('data', $data);
  }

  /**
   * Load the given plugin.
   */
  public function setPlugin($id) {
    $this->plugin = $this->csvImporter->createInstance($id);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Remember the form settings.
    $_SESSION['import']['format'] = $form_state->getValue('format');
    $rows = $this->plugin->getCsvRows($form_state->get('data'));
    $batch = $this->plugin->makeBatch($rows, $form_state->getValue('delete'), $form_state->getValue('test'));
    batch_set($batch);
    // Tidy up.
    if ($form_state->getValue('format') == 'file') {
      $form_state->getValue($form_state->getValue('import_type') . '_file')->delete();
    }
  }

}
