<?php

namespace Drupal\cforge_import\Plugin\CsvParser;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Plugin\PluginBase;
use Drupal\user\Entity\User;

ini_set('max_execution_time', 150);

/**
 * Base class for importing entities from csv files.
 */
abstract class ImportBase extends PluginBase implements CsvParserInterface {

  /**
   * {@inheritdoc}
   */
  public static function buildEntity(array $fields) {
    $entity = static::createEntity();
    foreach ($fields as $fieldname => $value) {
      $error = '';
      $class = get_called_class();
      // Fields with multiple columns.
      if (strpos($fieldname, '.')) {
        list($field, $col) = explode('.', $fieldname);
        $method = $field . '_' . $col . 'Process';
        if (method_exists($class, $method)) {
          $error = $class::{$method}($entity, $fields[$fieldname], $fields);
        }
        elseif ($value) {
          $entity->{$field}->{$col} = $value;
        }
      }
      // Time fields.
      elseif (in_array($fieldname, ['created', 'changed', 'access', 'expires'])) {
        $class::timeProcess($entity, $fieldname, $fields[$fieldname]);
      }
      else {
        $method = $fieldname . 'Process';
        if (method_exists($class, $method)) {
          $error = $class::$method($entity, $fields[$fieldname], $fields);
        }
        elseif ($entity->hasField($fieldname)) {
          $entity->set($fieldname, $value);
        }
        else {
          drupal_set_message("Field $fieldname does not exist on the entity", 'warning');
        }
      }
      if ($error) {
        drupal_set_message($error, 'warning');
      }
    }
    return $entity;
  }

  /**
   * {@inheritdoc}
   *
   * The sandbox seems to be empty every time, unfortunately.
   */
  public static function saveEntities($rows, $test, &$sandbox) {
    \Drupal::service('account_switcher')->switchTo(User::load(1));
    \Drupal::logger('Cforge')
      ->debug('Importing @count entities', ['@count' => count($rows)]);
    foreach ($rows as $row) {
      static::saveEntity($row, $test);
    }
    \Drupal::service('account_switcher')->switchBack();
  }

  /**
   * {@inheritdoc}
   */
  public static function saveEntity($row, $test = TRUE) {
    $entity = static::buildEntity($row);
    foreach ($entity->validate() as $violation) {
      \Drupal::logger('cforge')->error(
        "Failed validation: '@message' at @parents in <pre>@row</pre><br /><pre>@array</pre>",
        array(
          '@message' => $violation->getMessage(),
          '@parents' => $violation->getPropertyPath(),
          '@row' => print_r($row, 1),
          '@array' => print_r($entity->toArray(), 1),
        )
      );
      $test = TRUE;
    }
    if (!$test) {
      $result = $entity->save();
    }
    return $entity;
  }

  /**
   * {@inheritdoc}
   */
  public static function timeProcess(EntityInterface $account, $fieldname, $val) {
    if ($val) {
      if (!is_int($val)) {
        $val = strtotime($val);
      }
      $account->set($fieldname, $val);
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function langcodeProcess(EntityInterface $account, $val, $allfields) {
    $account->set('langcode', $val);
    $account->set('preferred_langcode', $val);
  }

  /**
   * {@inheritdoc}
   */
  public static function ready() {
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function makeBatch($rows, $delete = FALSE, $test = TRUE) {
    // This could be overwritten.
    $batch = ['title' => 'Creating entities'];
    if ($delete) {
      $batch['operations'][] = [[$this, 'deleteAll'], []];
    }
    foreach (array_chunk($rows, 25) as $chunk) {
      $batch['operations'][] = [[$this, 'saveEntities'], [$chunk, $test]];
    }
    return $batch;
  }

  /**
   * {@inheritdoc}
   */
  public function getCsvRows($data) {
    $csv = array_map('str_getcsv', explode(PHP_EOL, trim($data)));
    array_walk($csv, function(&$a) use ($csv) {
      if (count($csv[0]) != count($a)) {
        print_r($csv[0]);
        print_r($a);
      }
      $a = array_combine($csv[0], $a);
    });
    // Remove first row.
    array_shift($csv);
    $output = [];
    foreach ($csv as $key => $row) {
      foreach ($this->columns() as $col_name => $description) {
        if (isset($row[$col_name])) {
          $output[$key][$col_name] = $row[$col_name];
        }
      }
    }
    return $output;
  }

}
