<?php

namespace Drupal\cforge_import\Plugin\CsvParser;

use Drupal\smallads\Entity\Smallad;
use Drupal\file\Entity\File;
use Drupal\taxonomy\Entity\Term;
use Drupal\field\Entity\FieldConfig;
use Drupal\smallads\Entity\SmallAdInterface;

/**
 * Base class for importing smallads from csv file.
 */
abstract class Smallads extends ImportBase {

  /**
   * {@inheritdoc}
   */
  public static function deleteAll() {
    $entities = \Drupal::entityTypeManager()
      ->getStorage('smallad')
      ->loadByProperties(['type' => static::BUNDLE]);
    foreach ($entities as $entity) {
      $entity->delete();
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function ready() {
    // Can only import if there is more than one user.
    return \Drupal::entityQuery('user')->count()->execute() > 1;
  }

  /**
   * {@inheritdoc}
   */
  public static function createEntity() {
    return \Drupal::entityTypeManager()
      ->getStorage('smallad')
      ->create(['type' => static::BUNDLE]);
  }

  /**
   * {@inheritdoc}
   */
  public static function columns() {
    return [
      'name' => t('unique login name of owner user'),
      'title' => t('title of ad'),
      'body' => t('body of ad, in html'),
      'created' => t('item created date, format:  "2016-01-15 19:10:33'),
      'changed' => t('last changed date, format:  "2016-01-15 19:10:33"'),
      'scope' => t('0 means just the owner, 2 means every authenticated user can see it'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function makeBatch($rows, $delete = FALSE, $test = TRUE) {
    foreach (FieldConfig::loadMultiple() as $field) {
      if ($field->getTargetEntityTypeId() == 'smallad' && $field->getName() == 'image') {
        $dir = 'public://' . $field->getSetting('file_directory');
        if (!file_exists($dir)) {
          // Deprecated but \Drupal::service('file_system')->mkdir() didnt work.
          drupal_mkdir($dir, 509, 8);
          drupal_set_message("Created directory $dir");
        }
      }
    }
    $batch = parent::makeBatch($rows, $delete, $test);
    $batch['title'] = 'Importing ' . $this->getPluginDefinition()['label'];
    return $batch;
  }

  /**
   * Preprocessing for csv field 'body'.
   */
  public static function bodyProcess(SmallAdInterface $entity, $val, $allfields) {
    $entity->body->value = $val;
    $entity->body->format = 'basic';
  }

  /**
   * Preprocessing for csv field 'uid'.
   */
  public static function uidProcess(SmallAdInterface $entity, $val, $allfields) {
    $entity->setOwnerId($val);
  }

  /**
   * Preprocessing for csv field 'categories'.
   */
  public static function categoriesProcess(SmallAdInterface $entity, $val, $allfields) {
    if ($vals = array_filter(explode(',', $val))) {
      
      $existing_terms = \Drupal::entityTypeManager()
        ->getStorage('taxonomy_term')
        ->loadByProperties(['name' => $vals, 'vid' => SMALLAD_CATEGORIES]);
      // Key the existing terms by name so we can look them up.
      foreach ($existing_terms as $term) {
        $terms[$term->label()] = $term;
      }

      $categories = $entity->categories->getValue();
      foreach ($vals as $term_name) {
        $term_name = trim($term_name);
        if (!isset($terms[$term_name])) {
          $term = Term::Create(['name' => $term_name, 'vid' => SMALLAD_CATEGORIES]);
          $term->save();
          drupal_set_message('Created new term ' . $term_name);
          $terms[$term_name] = $term;
        }
        $categories[]['target_id'] = $terms[$term_name]->id();
      }
      $entity->categories->setvalue($categories);
    }
  }

  /**
   * Preprocessing for csv field 'picture'.
   */
  public static function pictureProcess(SmallAdInterface $entity, $val, $allfields) {
    if ($val) {
      $field = FieldConfig::load('smallad.' . static::BUNDLE . '.image');
      $newpath = 'public://'
        . $field->getSetting('file_directory') . '/'
        . pathinfo($val)['basename'];
      if (!file_exists($newpath)) {
        if (!copy($val, $newpath)) {
          drupal_set_message('Small ad not presaved. Unable to copy image to ' . $newpath);
          return;
        }
      }
      $file = File::create(['uri' => $newpath]);
      $file->save();
      $entity->set('image', $file);
      $entity->save();
    }
  }

  /**
   * Preprocessing for csv field 'name'.
   */
  public static function nameProcess(SmallAdInterface $entity, $val, $allfields) {
    $uids = \Drupal::entityQuery('user')->condition('name', $val)->execute();
    if (!$uids) {
      drupal_set_message('User not identified for name ' . $val);
      return;
    }
    $entity->setOwnerId(reset($uids));
  }

  /**
   * Preprocessing for csv field 'scope'.
   */
  public static function scopeProcess(SmallAdInterface $entity, $val, $allfields) {
    $entity->scope->value = $val ? Smallad::SCOPE_SITE : Smallad::SCOPE_PRIVATE;
  }

}
