<?php

namespace Drupal\cforge_import\Plugin\CsvParser;
use Drupal\mcapi\Entity\Transaction;
use Drupal\user\Entity\User;
use Drupal\mcapi\Entity\Currency;
use Drupal\mcapi\Entity\TransactionInterface;

/**
 * Import transactions from csv file.
 */
abstract class Transactions extends ImportBase {

  private $allWallets = [];

  /**
   * {@inheritdoc}
   */
  public static function columns() {
    return [
      'serial' => t('numeric identifer, not database key'),
      'payer' => t('integer wallet id e.g. abcd1234'),
      'payee' => t('integer wallet id e.g. abcd1234'),
      'quantity' => t('integer value of transaction'),
      'created' => t('Item created date in any strtotime format'),
      'state' => t('is always 1 which translates to transaction state "done" in D8'),
      'type' => t('string transaction type. name of Type plugin e.g. default'),
      'description' => t('plain text'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function deleteAll() {
    foreach (\Drupal::entityTypeManager()->getStorage('mcapi_transaction')->loadMultiple() as $entity) {
      $entity->delete();
    }
  }

  /**
   * Preprocessing for csv field 'state'.
   */
  protected static function stateProcess(TransactionInterface $entity, $val, $allfields) {
    if (is_numeric($val)) {
      switch ($val) {
        case 1: $state = 'done';
          break;

        case 0: $state = 'erased';
          break;

        case -1: $state = 'pending';
          break;

        default:
          throw new \Exception('Unknown transaction state: ' . $val);
      }
    }
    elseif (is_string($val)) {
      $state = $val;
    }
    $entity->state->target_id  = 'done';
  }

  /**
   * Preprocessing for csv field 'type'.
   */
  protected static function typeProcess(TransactionInterface $entity, $val, $allfields) {
    switch ($val) {
      case '3rdparty':
        break;

      case 'auto':
      case 'remote':
        break;

      case '1stparty':
      case 'default':
      default:
        $type = 'default';
    }
    $entity->type->target_id  = strtolower($type);
  }

  /**
   * Preprocessing for csv field 'body'.
   */
  protected static function quantityProcess(TransactionInterface $entity, $val, $allfields) {
    $currency = Currency::load('cc');
    @list($talents, $cents) = explode('.', $val);
    $entity->worth->curr_id = 'cc';
    $entity->worth->value = $currency->unformat([1 => $talents, 3 => $cents]);
  }

  /**
   * {@inheritdoc}
   */
  public static function saveEntities($rows, $test, &$sandbox) {
    \Drupal::service('account_switcher')->switchTo(User::load(1));
    foreach ($rows as $rownum => $row) {
      $entity = static::buildEntity($row);
      foreach ($entity->validate() as $violation) {
        drupal_set_message($violation->getMessage());
      }
      if (!$test) {
        $class = get_called_class();
        $entity = $class::buildEntity($row);
        Self::stash($entity, $sandbox);
      }
    }
    \Drupal::service('account_switcher')->switchBack();
  }

  /**
   * Stash a transaction in the tempStore.
   */
  public static function stash($transaction, &$sandbox) {
    $tempstore = \Drupal::service('user.private_tempstore')->get('saved');
    // Get the previous transaction if there is one, save it, and put the
    // current transaction in the tempstore.
    $previous_transaction = $tempstore->get('stash');
    if ($previous_transaction instanceof Transaction) {
      if ($transaction->type->target_id == 'default') {
        $previous_transaction->save();
        \Drupal::logger('cforge_import')->debug(
          'Imported transaction @serial with @count children: @print',
          [
            '@serial' => $previous_transaction->serial->value, 
            '@count' => count($previous_transaction->children), 
            '@print' => print_r($previoustransaction->toArray())
          ]
        );
        $tempstore->set('stash', $transaction);
      }
      elseif ($transaction->type->target_id == 'auto') {
        $previous_transaction->children[] = $transaction;
        $tempstore->set('stash', $previous_transaction);
        \Drupal::logger('cforge_import')->debug('Stashed transaction');
      }
    }
    // First time we run this put the current transaction in the tempstore.
    else {
      $tempstore->set('stash', $transaction);
    }
  }

  /**
   * Get the last transaction from the stash and save it.
   *
   * Used only at the end of the batch.
   */
  public static function saveLastTransaction($success, $results, $operations) {
    // Trigger saving the last transaction
    // ugly workaround because the sandbox appears not to be working.
    $last = \Drupal::service('user.private_tempstore')->get('saved')->get('stash');
    if ($last instanceof Transaction) {
      $last->save();
    }
    else {
      drupal_set_message('problem saving last transaction: ' . gettype($last));
    }
    \Drupal::service('user.private_tempstore')->get('saved')->delete('stash');
  }

  /**
   * {@inheritdoc}
   */
  public function makeBatch($rows, $delete = FALSE, $test = TRUE) {
    // @todo this should really be injected
    \Drupal::service('user.private_tempstore')->get('saved')->delete('stash');
    $batch = parent::makeBatch($rows, $delete, $test);
    $batch['title'] = 'Importing transaction history';
    if (!$test) {
      $batch['finished'] = [$this, 'saveLastTransaction'];
    }
    return $batch;
  }

  /**
   * {@inheritdoc}
   */
  public static function createEntity() {
    return \Drupal::entityTypeManager()
      ->getStorage('mcapi_transaction')
      ->create(['type' => 'default']);
  }

}
