<?php

namespace Drupal\cforge_import\Plugin\CsvParser;

use Drupal\file\Entity\File;
use \Drupal\node\Entity\Node;
use Drupal\field\Entity\FieldConfig;
use Drupal\user\UserInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;

/**
 * Base plugin for importing users.
 */
abstract class Users extends ImportBase implements ContainerFactoryPluginInterface {

  protected $siteSettings;

  /**
   * Constructor.
   */
  public function __construct($configuration, $plugin_id, $plugin_definition, $config_factory) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->siteSettings = $config_factory->get('system_site');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function deleteAll() {
    foreach (\Drupal::entityTypeManager()->getStorage('user')->loadMultiple() as $entity) {
      if ($entity->id() < 2) {
        // don't delete user 1.
        continue;
      }
      $entity->delete();
    }
  }

  /**
   * {@inheritdoc}
   */
  protected static function createEntity() {
    return \Drupal::entityTypeManager()->getStorage('user')->create()->activate();
  }

  /**
   * {@inheritdoc}
   */
  public static function columns() {
    return [
      'name' => t('user/login name'),
      'mail' => t('email address*'),
      'password' => t('password'),
      'roles' => t('name of the role, e.g. trader, committee'),
      'firstname' => t('first name*'),
      'lastname' => t('last name'),
      'created' => t('created date in any parsable php strtotime format'),
      'access' => t('last login time in any parsable php strtotime format'),
      'user_picture' => t('absolute url of image file'),
      'address.address_line1' => t('address line 1'),
      'address.address_line2' => t('address line 2'),
      'address.organization' => t('organisation name'),
      'address.dependent_locality' => t('Neighbourhood name (N.B references node type.*'),
      'address.locality' => t('City/town'),
      'address.administrative_area' => t('Region/Province'),
      // 'address.postal_code', we have  no way to force these to be valid.
      'address.country_code' => t('2 digit country code (default is @default)', ['@default' => \Drupal::config('system.date')->get('country.default')]),
      'phones' => t('mobile phone number'),
      'phones2' => t('other phone number'),
      'notes' => t("'about me' text in html"),
      'notes_admin' => t('plaintext notes visible only to committee'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function makeBatch($rows, $delete = FALSE, $test = TRUE) {
    $field = FieldConfig::load('user.user.user_picture');
    $dir = 'public://' . $field->getSetting('file_directory');
    // Deprecated but \Drupal::service('file_system')->mkdir($dir) didn't work.
    // Suppress the error if it doesn't work
    @drupal_mkdir($dir, 509, 8);

    $batch = parent::makeBatch($rows, $delete, $test);
    $batch['title'] = 'Importing users';
    return $batch;
  }

  /**
   * Preprocessing for csv field 'name'.
   */
  public static function nameProcess(UserInterface $account, $val, $allfields) {
    $account->setUsername($val);
  }

  /**
   * Preprocessing for csv field 'firstname'.
   */
  public static function firstnameProcess(UserInterface $account, $val, $allfields) {
    if (static::required($val)) {
      $account->address->given_name = $val;
    }
    else {
      return 'missing firstname';
    }
  }
  /**
   * Preprocessing for csv field 'firstname'.
   */
  public static function lastnameProcess(UserInterface $account, $val, $allfields) {
    $account->address->family_name = $val;
  }

  /**
   * Preprocessing for csv field 'password'.
   */
  public static function passwordProcess(UserInterface $account, $val, $allfields) {
    $account->setPassword($val);
  }

  /**
   * Preprocessing for csv field 'mail'.
   */
  public static function mailProcess(UserInterface $account, $val, $allfields) {
    if (empty($val) or !preg_match('/^.+\@\S+\.\S+$/', $val)) {
      $sitemail = \Drupal::config('system.site')->get('mail');
      $suffix = substr($sitemail, strpos($sitemail, '@'));
      if (!$suffix) {
        throw new Exception('Site mail cannot be used for creating dummy emails: ' . $sitemail);
      }
      $val = $allfields['name'] . $suffix;
    }
    $account->setEmail($val);
    $account->set('init', $val);
  }

  /**
   * Preprocessing for csv field 'coordinator'.
   */
  public static function coordinatorProcess(UserInterface $account, $val, &$allfields) {
    $account->addRole('superneighbour');
  }

  /**
   * Preprocessing for csv field 'address_dependent_locality'.
   *
   * @note Eventually this field will be a group and thus the address will
   * reference it.
   */
  public static function address_dependent_localityProcess(UserInterface $account, $val, $allfields) {
    static::required($val, 'Brisbane');
    if (!\Drupal::entityQuery('node')->condition('type', 'neighbourhood')->condition('title', $val)->execute()) {
      Node::create(['type' => 'neighbourhood', 'title' => $val])->save();
    }
    $account->address->dependent_locality = $val;
  }

  /**
   * Preprocessing for csv field 'phones2'.
   */
  public static function phones2Process(UserInterface $account, $val, $allfields) {
    $vals = [$allfields['phones'], $val];
    $account->set('phones', $vals);
  }

  /**
   * Preprocessing for csv field 'notes'.
   */
  public static function notesProcess(UserInterface $account, $val, $allfields) {
    $val = str_replace('<br />', '\n', $val);
    $account->set('notes', $val);
  }

  /**
   * Preprocessing for csv field 'user_picture'.
   */
  public static function user_pictureProcess(UserInterface $account, $src, $allfields) {
    if ($src) {
      $info = pathinfo($src);
      $newpath = 'private://'
        . FieldConfig::load('user.user.user_picture')->getSetting('file_directory') . '/'
        . pathinfo($src)['basename'];
      if (!file_exists($newpath)) {
        if (!copy(rawurlencode($src), $newpath)) {
          drupal_set_message('User not presaved. Unable to copy image to ' . $newpath);
          return;
        }
      }
      $file = File::create(['uri' => $newpath, 'uid' => $account->id()]);
      $file->save();
      $account->user_picture = $file;
      // This has to be saved before validation entity reference refuses access
      // before file_usage is written
      // Drupal\Core\Field\EntityReferenceFieldItemList/FileFieldItemList::postsave()
      $account->save();
    }
  }

  /**
   * The sandbox seems to be empty every time, unfortunately.
   */
  public static function saveEntities($rows, $test, &$sandbox) {
    $default_country = \Drupal::config('system.date')->get('country.default');
    foreach ($rows as $key => &$row) {
      if (empty($row['password'])) {
        unset($rows[$key]);
        // Transactions with unknown wallet ids will have a new wallet created.
      }
      elseif (empty($row['address.country_code'])) {
        $row['address.country_code'] = $default_country;
      }
    }
    parent::saveEntities($rows, $test, $sandbox);
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Exception
   */
  public static function required($value, $default = NULL) {
    if ($value) {
      return $value;
    }
    if ($default) {
      return $default;
    }
    throw new \Exception('null required value');
  }

}
