/**
 * @file
 * A Backbone View that adds "Add new item" functionality to widget containers.
 *
 * @see Drupal.shs.ContainerView
 */

(function ($, Backbone, Drupal) {

  'use strict';

  Drupal.shs.AddNewView = Backbone.View.extend(/** @lends Drupal.shs.AddNewView# */{
    /**
     * The main application.
     *
     * @type {Drupal.shs.AppView}
     */
    app: null,
    /**
     * Default tagname of this view.
     *
     * @type {string}
     */
    tagName: 'div',
    /**
     * @constructs
     *
     * @augments Backbone.View
     *
     * @param {object} options
     *   An object with the following keys:
     * @param {Drupal.shs.AppView} options.app
     *   The application state view.
     */
    initialize: function (options) {
      this.app = options.app;
      // Listen to collection updates.
      this.listenTo(this.app.collection, 'widgetsRendered:shs', this.addButton);
    },
    /**
     * Add button to application if needed.
     *
     * @return {Drupal.shs.AddNewView}
     *   Returns AddNewView for chaining.
     */
    addButton: function () {
      var element = this;

      // Remove buttons created earlier.
      $('.shs-addnew-container', element.app.container).remove();

      // Set default classes and clear content.
      element.$el.addClass('shs-addnew-container')
              .html('');

      // Does the setting allow us to add more items?
      var cardinality = this.app.getConfig('cardinality');
      var itemCount = this.app.collection.length;

      if ((cardinality === 1) || (cardinality === itemCount)) {
        // Only 1 item is allowed or we reached the maximum number of items.
        return;
      }

      // Create "button".
      var $button = $('<a>')
              .addClass('button')
              .addClass('add-another')
              .text(this.app.getSetting('addNewLabel'));

      $button.on('click', {app: this.app, button: this}, this.triggerContainerRefresh);
      $button.appendTo(element.$el);

      // Add element to application container.
      element.$el.appendTo(element.app.container);

      // Return self for chaining.
      return element;
    },
    /**
     * Trigger a refresh of the application container.
     *
     * @param {object} event
     *   The event object containing the following event.data:
     * @param {object} event.data.button
     *   The current view.
     * @param {object} event.data.app
     *   Reference to the main application.
     */
    triggerContainerRefresh: function (event) {
      var element = event.data.button;
      var app = event.data.app;
      element.$el.addClass('is-disabled');
      app.collection.add(new Drupal.shs.classes[app.getConfig('fieldName')].models.container({
        delta: app.collection.length,
        parents: [{
          defaultValue: app.getSetting('anyValue'),
          parent: 0
        }]
      }));
      element.$el.removeClass('is-disabled');
      app.container.html('');
      app.collection.trigger('initialize:shs');
    }
  });

}(jQuery, Backbone, Drupal));
