<?php

namespace Drupal\smallads\Plugin\DevelGenerate;

use Drupal\Component\Utility\Random;
use Drupal\comment\Entity\Comment;
use Drupal\smallads\Entity\SmallAdInterface;
use Drupal\smallads\Entity\SmallAd;
use Drupal\smallads\Entity\SmallAdType;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\devel_generate\DevelGenerateBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a DevelGenerate plugin.
 *
 * @DevelGenerate(
 *   id = "smallad",
 *   label = @Translation("smallads"),
 *   description = @Translation("Generate a given number of smallads.."),
 *   url = "smallad",
 *   permission = "administer devel_generate",
 *   settings = {
 *     "num" = 100,
 *     "kill" = TRUE
 *   }
 * )
 */
class SmalladDevelGenerate extends DevelGenerateBase implements ContainerFactoryPluginInterface {

  const MAX = 100;

  /**
   * The smallad storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $smalladStorage;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  protected $uids;

  /**
   * {@inheritdoc}
   */
  public function __construct(array $configuration, $plugin_id, array $plugin_definition, $entity_type_manager, $module_handler) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->moduleHandler = $module_handler;
    $this->smalladStorage = $entity_type_manager->getStorage('smallad');
    $this->uids = $entity_type_manager->getStorage('user')->getQuery()->condition('status', TRUE)->execute();
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration, $plugin_id, $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('module_handler')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $form['kill'] = array(
      '#type' => 'checkbox',
      '#title' => $this->t('<strong>Delete all smallads</strong> before generating new content.'),
      '#default_value' => $this->getSetting('kill'),
    );
    $form['num'] = array(
      '#type' => 'number',
      '#title' => $this->t('How many smallads would you like to generate?'),
      '#default_value' => $this->getSetting('num'),
      '#required' => TRUE,
      '#min' => 0,
    );
    $types = array_keys(SmallAdType::loadMultiple());
    $form['type'] = array(
      '#title' => $this->t('Type'),
      '#type' => 'select',
      '#options' => ['' => $this->t('- All -')] + array_combine($types, $types),
      '#default_value' => $this->getSetting('type'),
      '#min' => 0,
    );
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  protected function generateElements(array $values) {
    if ($values['num'] <= static::MAX) {
      $this->generateContent($values);
    }
    else {
      $this->generateBatchContent($values);
    }
  }

  /**
   * Method responsible for creating a small number of smallads.
   *
   * @param array $values
   *   Kill, num, since.
   *
   * @throws \Exception
   */
  private function generateContent(array $values) {
    if (!empty($values['kill'])) {
      $this->contentKill($values['type']);
    }
    $this->since = isset($values['since']) ? $values['since'] : strtotime('-1 year');
    for ($i = 1; $i <= $values['num']; $i++) {
      $this->develGenerateSmalladAdd($values['type']);
    }
    if (function_exists('drush_log') && $i % drush_get_option('feedback', 1000) == 0) {
      drush_log(dt('Completed @feedback smallads ', ['@feedback' => drush_get_option('feedback', 1000)], 'ok'));
    }
    $this->setMessage(
      $this->formatPlural(
        $values['num'],
        '1 @type created.',
        'Created @count @types',
        ['@type' => $values['type'] ? SmallAdType::load($values['type'])->label() : $this->t('Small ad')]
      )
    );
  }

  /**
   * Method responsible for creating more than 50 items at a time.
   */
  private function generateBatchContent($values) {
    // Add the kill operation.
    if ($values['kill']) {
      $operations[] = [
        'devel_generate_operation',
        [$this, 'batchContentKill', $values['type']],
      ];
    }
    $total = $values['num'];
    $values['num'] =  static::MAX;
    // Add the operations to create the transactions.
    for ($num = 0; $num <= floor($total / static::MAX); $num++) {
      $operations[] = [
        'devel_generate_operation',
        [$this, 'batchContentAddSmallads', $values],
      ];
    }
    $values['num'] = $total % static::MAX;
    $operations[] = [
      'devel_generate_operation',
      [$this, 'batchContentAddSmallads', $values],
    ];

    // Start the batch.
    $batch = [
      'title' => $this->t('Generating Small ads'),
      'operations' => $operations,
      'finished' => 'devel_generate_batch_finished',
      'file' => drupal_get_path('module', 'devel_generate') . '/devel_generate.batch.inc',
    ];
    batch_set($batch);
  }

  /**
   * Batch callback.
   */
  public function batchContentAddSmallads($values, &$results) {
    $this->since = $values['since'];
    $i = 0;
    while ($i < $values['num']) {
      $this->develGenerateSmalladAdd($values['type']);
      $i++;
    }
    // Notice: Undefined index: num
    // $results['num'] += $values['num'];
    $this->setMessage(
      $this->formatPlural(
        $values['num'],
        '1 @type created.',
        'Created @count @types',
        [
          '@count' => $values['num'],
          '@type' => $values['type'] ? SmallAdType::load($values['type'])->label() : $this->t('Small ad')
        ]
      )
    );
  }

  /**
   * {@inheritdoc}
   */
  public function batchContentKill($type) {
    $this->contentKill($type);
  }

  /**
   * {@inheritdoc}
   */
  public function validateDrushParams($args) {
    $values['kill'] = drush_get_option('kill');
    $values['type'] = drush_get_option('type');
    $values['num'] = array_shift($args);
    return $values;
  }

  /**
   * Deletes all smallads of the given type
   *
   * @param array $type
   *   The input values from the settings form.
   */
  protected function contentKill($type = NULL) {
    $props = [];
    if ($type) {
      $props['type'] = $type;
    }
    $smallads = $this->smalladStorage->loadByProperties($props);
    if (!empty($smallads)) {
      $this->smalladStorage->delete($smallads);
      $this->setMessage(
        $this->t(
          'DevelGenerate deleted %count @types.',
          [
            '%count' => count($smallads),
            '@type' => $type ? SmallAdType::load($type)->label() : $this->t('Small ad')
          ]
        )
      );
    }
  }

  /**
   * Create one smallad. Used by both batch and non-batch code branches.
   */
  protected function develGenerateSmalladAdd($type = NULL) {
    if (!$type) {
      $types = array_keys(SmallAdType::loadMultiple());
      $type = $types[rand(0, count($types) - 1)];
    }
    $first = [
      'A very nice',
      'A nearly expired',
      'A French',
      'A traditional',
      'An organic',
      'Jewish',
      'Refurbished',
      'Beautiful',
      'Hand-crafted',
      'Antique',
      'High-powered',
      '', '',
    ];
    $second = [
      'porcelaine',
      'green',
      'unwanted',
      'licenced',
      'brown',
      'underwater',
      'fried',
      'stress-tested',
      'double-loaded',
      'ex-rental',
      'gelatinous',
    ];
    $third = ['dragon',
      'antique',
      'dolly',
      'buffet',
      'ballet lessons',
      'donkey',
      'ladder',
      'mp3 player',
      'widgets',
      'armistice',
      '', '',
    ];
    $fourth = [
      'from the orient.',
      'in need of repair.',
      'unwanted gift.',
      'in perfect condition.',
      'for hire.',
      'latest model!',
      '', '', '',
    ];


    $category_ids = \Drupal::entityQuery('taxonomy_term')
        ->condition('vid', 'categories')
        ->execute();
    shuffle($category_ids);
    $props = [
      'type' => $type,
      'title' => $first[array_rand($first)] . ' ' . $second[array_rand($second)] . ' ' . $third[array_rand($third)] . ' ' . $fourth[array_rand($fourth)],
      'body' => $this->getRandom()->paragraphs(2),
      'categories' => array_slice($category_ids, 0, rand(1,2)),
      'uid' => $this->randomUid(),
      'scope'  => array_rand(SmallAd::getScopes()),
      'directexchange' => rand(0, 1),
      'indirectexchange' => rand(0, 1),
      'money' => rand(0, 1),
    ];
    $smallad = SmallAd::create($props);
    $smallad->created->value = rand($this->since, REQUEST_TIME);
    // Populate all fields with sample values.
    $this->populateFields($smallad);
    $smallad->save();
    if ($smallad->getFieldDefinition('comments')) {
      $this->addComments($smallad);
    }
  }

  /**
   * Get a random user uid.
   */
  private function randomUid() {
    return $this->uids[array_rand($this->uids)];
  }

  /**
   * Create comments and add them to a smallad.
   *
   * @param SmallAdInterface $smallad
   *   Smallad to add comments to.
   */
  public function addComments(SmallAdInterface $smallad) {
    $parents = array();
    $num_comments = mt_rand(1, 3);
    for ($i = 1; $i <= $num_comments; $i++) {
      switch ($i % 3) {
        case 0:
          // No parent.
        case 1:
          // Top level parent.
          $parents = \Drupal::entityQuery('comment')
            ->condition('pid', 0)
            ->condition('entity_id', $smallad->id())
            ->condition('entity_type', 'smallad')
            ->condition('field_name', 'comments')
            ->range(0, 1)
            ->execute();
          break;

        case 2:
          // Non top level parent.
          $parents = \Drupal::entityQuery('comment')
            ->condition('pid', 0, '>')
            ->condition('entity_id', $smallad->id())
            ->condition('entity_type', 'smallad')
            ->condition('field_name', 'comments')
            ->range(0, 1)
            ->execute();
          break;
      }
      $random = new Random();
      $stub = array(
        'entity_type' => 'smallad',
        'entity_id' => $smallad->id(),
        'comment_type' => 'smallad',
        'field_name' => 'comments',
        'created' => mt_rand($smallad->get('created')->value, REQUEST_TIME),
        'subject' => substr($random->sentences(mt_rand(2, 6), TRUE), 0, 63),
        'langcode' => $smallad->language()->getId(),
      );
      if ($parents) {
        $stub['pid'] = current($parents);
      }
      $comment = Comment::create($stub);

      // Populate all core fields on behalf of field.module.
      DevelGenerateBase::populateFields($comment);
      $comment->uid = $this->randomUid();
      $comment->setFieldname('comments');
      $comment->entity_type->value = 'smallad';
      $comment->entity_id->value = $smallad->id();
      $comment->save();
    }
  }

}
