<?php

namespace Drupal\cforge\EventSubscriber;

use Drupal\address\Event\AddressFormatEvent;
use Drupal\Core\Url;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpKernel\KernelEvents;
use Symfony\Component\HttpKernel\Event\GetResponseEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Cforge react to events.
 */
class CforgeSubscriber implements EventSubscriberInterface {

  private $currentUser;
  private $cforgeSettings;
  private $routeMatch;
  private $moduleHandler;

  function __construct($current_user, $config, $route_match, $module_handler) {
    $this->currentUser = $current_user;
    $this->cforgeSettings = $config->get('cforge.settings');
    $this->routeMatch = $route_match;
    $this->moduleHandler = $module_handler;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    return [
       KernelEvents::REQUEST => [['cforgeRedirect']],
       // For some reasong Drupal\address\Event\AddressEvents::ADDRESS_FORMAT can't be seen from here
       'address.address_format' => [['rewriteAddress']]
    ];
  }


  /**
   * Redirect users
   */
  public function cforgeRedirect(GetResponseEvent $event) {
    if ($this->currentUser->isAuthenticated()) {
//      \Drupal::configFactory()->getEditable('cforge.settings')->set('flavour', 'hamlets')->save();
      if ($this->cforgeSettings->get('flavour')) {
        if (!$this->moduleHandler->moduleExists('migrate')) {
          $route_name = $this->routeMatch->getRouteName();
          //these are all routes which can be invoked BEFORE cforge.setup
          $dont_divert = [
            'setup',
            'user.page',
            'system',
            'admin',
            'overview',
            'collection',
            'migrate_drupal_ui.upgrade'
          ];
          foreach ($dont_divert as $rname) {
            $pos = strpos($route_name, $rname);
            if (is_integer($pos)) return;
          }
          drupal_set_message('Diverted from '.$route_name .' with flavour: '.$this->cforgeSettings->get('flavour'));
          // Redirect to the setup wizard
          $response = new RedirectResponse(Url::fromRoute('cforge.setup')->toString());
          $event->setResponse($response);

        }
      }
      // Different front page for anon and authenticated users.
      elseif ($event->getRequest()->getpathinfo() == '/') {
        $response = new RedirectResponse($this->cforgeSettings->get('member_frontpage'));
        $event->setResponse($response);
      }
    }
    else {
      $path = trim($event->getRequest()->getpathinfo(), '/');
      // If there was a trailing slash filter the empty part.
      $parts = array_filter(explode('/', $path));
      if (count($parts) > 5) {
        \Drupal::logger('cforge')->notice("Blocked access to path %path", ['%path' => $path]);
        // This was originally added to deter certain kinds of bots probing long paths
        // @todo this exception is uncaught when anon tries to access an image style, say in an event.
        //throw new RouteNotFoundException('Access denied');
      }
    }
  }


  /**
   * Ensure every address format has %dependentLocality, sticking it on the end
   * if necessary, that that field is required, and called neighbourhood
   */
  public function rewriteAddress(AddressFormatEvent $event) {
    $addressFormat = $event->getDefinition();
    $pattern = $addressFormat['format'];
    if (!strpos($pattern, '%dependentLocality')) {
      $pattern .= "\n%dependentLocality";
      $addressFormat['format'] = $pattern;
    }
    $addressFormat['dependent_locality_type'] = 'neighborhood';
    $addressFormat['required_fields'] = ['dependentLocality', 'givenName'];
    $event->setDefinition($addressFormat);
  }

}
