<?php

namespace Drupal\cforge\EventSubscriber;

use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Drupal\migrate\MigrateSkipRowException;
use Drupal\migrate\Event\MigratePreRowSaveEvent;
use Drupal\migrate\Event\MigrateImportEvent;
use Drupal\migrate\Row;
use Drupal\field\Entity\FieldConfig;
use Drupal\Core\Entity\Entity\EntityFormDisplay;
use Drupal\Core\Entity\Entity\EntityViewDisplay;
use Drupal\Core\Extension\ModuleHandler;

/**
 * Migration modifications
 */
class MigrationSubscriber implements EventSubscriberInterface {

  /**
   * @var Drupal\Core\Extension\ModuleHandler
   */
  private $moduleHandler;

  const FILTER_FORMATS = [
    'editor_full_html' => 'full_html',
    'editor_filtered_html' => 'basic',
    'filtered_html' => 'basic',
    'full_html' => 'full_html',
    'plain_text' => '',
    ''  => 'full_html'
  ];

  function __construct(ModuleHandler $module_handler) {
    $this->moduleHandler = $module_handler;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    return [
      \Drupal\Core\Config\ConfigEvents::SAVE => 'onSaveConfig',//temp
      //Drupal\migrate\Event\MigrateEvents::PRE_IMPORT
      'migrate.pre_import' => [['migratePreImport']],
      //Drupal\migrate\Event\MigrateEvents::PRE_ROW_SAVE
      'migrate.pre_row_save' => [['migratePreRowSave']]
    ];
  }

  /**
   * @param Drupal\migrate\Event\MigrateImportEvent $event
   */
  public function migratePreImport(MigrateImportEvent $event) {
    $migration = $event->getMigration();
    if ($migration->id() == 'd7_field_instance') {
      // Find any duplicate field instances
      $db = $migration->getSourcePlugin()->getDatabase();
      $dupes = $db->query("SELECT min(id) FROM {field_config_instance} WHERE deleted = 0 GROUP BY entity_type, field_name, bundle HAVING count(id) > 1")->fetchCol();
      if ($dupes) {
        // Delete them from the ORIGINAL database
        $db->delete('field_config_instance')->condition('id', $dupes, 'IN')->execute();
      }
    }
  }


  /**
   * @param Drupal\migrate\Event\MigratePreRowSaveEvent $event
   */
  public function migratePreRowSave(MigratePreRowSaveEvent $event) {
    $row = $event->getRow();
    $migration = $event->getMigration();

    // Ensure the category->required setting is transferred.
    if ($migration->id() == 'd7_field_instance' and $row->getSourceProperty('field_name') == 'offers_wants_categories' and $row->getSourceProperty('entity_type') == 'transaction') {
      // Just take this one property which varied in some d7 sites then ditch the field
      $field = FieldConfig::load('mcapi_transaction.mcapi_transaction.category');
      $field->setRequired($row->getSourceProperty('required'))->save();
    }
    if ($migration->id() == 'd7_field' or $migration->id() == 'd7_field_instance') {
      $field_name = $row->getSourceProperty('field_name');
      //Skip these d7 fields and rename them when importing the entities they belong to.
      $existing_fields = [
        'body',
        'cforge_docs_categories',
        'event_date',
        'galleries',
        'image',
        'manage_notes',
        'offers_wants_categories',
        'offers_wants_types',
        'profile_notes',
        'profile_phones',
        'profile_address',
        'profile_location',
        'upload', //became 'attached'
        'upload_private', // Became 'attached',
        //'title', // Prevents 'base_field_override' fields being created
      ];
      if (in_array($field_name, $existing_fields)) {
        throw new MigrateSkipRowException("Field $field_name already exists in Hamlets for Drupal 8");
      }
    }

    // All novel user fields should go on the 'profile' form display and 'default' display
    if ($migration->id() == 'd7_field_instance' and $row->getSourceProperty('entity_type') == 'user') {
      $formDisplay = EntityFormDisplay::load('user.user.profile');
      $formDisplay->setComponent($row->getSourceProperty('field_name'), ['weight' => 20])->save();
      $viewDisplay = EntityViewDisplay::load('user.user.default');
      $viewDisplay->setComponent($row->getSourceProperty('field_name'), ['weight' => 20])->save();
    }

    // Field instances showing on entity form displays
    if ($migration->id() == 'd7_field_instance_widget_settings') {
      if ($row->getSourceProperty('field_name') == 'profile_address') {
        // The addressfield available_countries and default country are stored in
        // the d7 field config instance widget setting.
        // Needs to go to the user.user.address field setting
        $settings = $row->getSourceProperty('widget')['settings'];
        FieldConfig::load('user.user.address')
          ->setSetting('available_countries', $settings['available_countries'])
          ->setSetting('default_country', $settings['default_country'])
          ->save();
        throw new MigrateSkipRowException("Migrated widget settings only from user address widget");
      }
      else {
        // Ensure any fields which have been added by user 1 are visible on the profile tab.
        // Don't bother about the weight
        $row->setDestinationProperty('form_mode', 'profile');
      }
    }


    if($migration->id() == 'd7_user') {
      // Rename the fields
      $map = [
        'manage_notes' => 'notes_admin',
        'profile_notes' => 'notes',
        'profile_phones' => 'phones',
        'profile_address' => 'address',
        'profile_loccation' => 'coordinates'
      ];
      foreach ($map as $old_name => $new_name) {
        if ($row->hasDestinationProperty($old_name)) {
          $row->setDestinationProperty($new_name, $row->getDestinationProperty($old_name));
          $row->removeDestinationProperty($old_name);
        }
      }
    }

    // Move all terms that used to be in offers_wants_categories field.
    if($migration->id() == 'd7_node' or $migration->id() == 'd7_mcapi_transaction') {
      if ($row->hasDestinationProperty('offers_wants_categories')) {
        $entity_type = $row->getSourceProperty('entity_type');
        $new_field_name = $row->getSourceProperty('entity_type') == 'node' ? 'categories' : 'category';
        $row->setDestinationProperty($new_field_name, $row->getSourceProperty('offers_wants_categories'));
        $row->removeDestinationProperty('offers_wants_categories');
      }
    }
    // Change the filter format for every body field
    // I don't know why this doesn't happen already
    if($row->getSourceProperty('plugin') == 'd7_node') {
      $this->entityBodyFilterFormat($row, 'body');

      // Change the name of the image taxonomy field
      if ($migration->id() == 'd7_node:image') {
        $row->setDestinationProperty('gallery', $row->getDestinationProperty('galleries'));
        $row->removeDestinationProperty('galleries');
      }
      // Change the name of the document taxonomy field
      if ($migration->id() == 'd7_node:document') {
        $row->setDestinationProperty('doctype', $row->getDestinationProperty('cforge_docs_categories'));
        $row->removeDestinationProperty('cforge_docs_categories');
      }
    }

    if($row->getSourceProperty('plugin') == 'd7_comment') {
      print_r($row->getDestination());
      $this->entityBodyFilterFormat($row, 'comment_body');
      print_r($row->getDestination());
    }
    if($row->getSourceProperty('plugin') == 'd7_block_content') {
      print_r($row->getDestination());
      $this->entityBodyFilterFormat($row, 'body');
      print_r($row->getDestination());
    }

    if ($migration->id() == 'd7_taxonomy_vocabulary') {
      $vid = $row->getSourceProperty('machine_name');
      if (in_array($vid, ['galleries', 'cforge_docs_categories'])) {
        throw new MigrateSkipRowException("Vocabluary $vid already exists in Hamlets for Drupal 8");
      }
    }
    // Move the terms to the new gallery name
    elseif ($migration->id() == 'd7_taxonomy_term:galleries') {
      $row->setDestinationProperty('vid', 'galleries');
    }
    elseif ($migration->id() == 'd7_taxonomy_term:cforge_docs_categories') {
      $row->setDestinationProperty('vid', 'binders');
    }
    elseif($migration->id() == 'd7_comment' and $row->getDestinationProperty('entity_type') != 'smallad') {
      $row->setDestinationProperty('comment_type', $row->getDestinationProperty('entity_type'));
      $row->setDestinationProperty('field_name', 'comments');
      $row->setDestinationProperty('body', $row->getDestinationProperty('comment_body'));// Assumes no comments on transactions
    }
    elseif ($migration->id() == 'contact_category') {
       //echo "\nChecking contact import:"; print_r($row->getDestination());
    }
    elseif ($migration->id() == 'd7_action') {
       $callback = $row->getSourceProperty('callback');
       if (substr($callback, 0, 22) == 'views_bulk_operations_') {
         if (!$this->moduleHandler()->moduleExists('views_bulk_operations')) {
           throw new MigrateSkipRowException('Views Bulk Operations module not installed.');
         }
       }
       elseif (!is_callable($callback)) {
         throw new MigrateSkipRowException('Callback not found: '.$callback);
       }
    }
  }

  /**
   *
   * @deprecated
   */
  private function entityBodyFilterFormat(Row $row, $body_field_name) {
    if ($bods = $row->getDestinationProperty($body_field_name)) {
      $map = static::FILTER_FORMATS;
      foreach ($bods as $delta => &$body) {
        if ($new_format = $map[$body['format']]) {
          $bods[$delta]['format'] = $new_format;
        }
        else {
          \Drupal::logger('Filter Format migration')->error(
            'Could not migrate filter format on @field: @row',
            ['@field' => $body_field_name, '@row' => print_r($row->getSource(), 1)]
          );
        }
      }
      $row->setDestinationProperty($body_field_name, $bods);
    }
  }

  function onSaveConfig(\Drupal\Core\Config\ConfigCrudEvent $event) {
    $config = $event->getConfig();
    if ($config->getName() == 'sky.settings') {
      mtrace(['args', 'log']);
    }
    if ($config->getName() == 'cfbel.settings') {
      mtrace(['args', 'log']);
    }
  }

}
