<?php

namespace Drupal\cforge\Forms;

use Drupal\Core\Cache\Cache;
use Drupal\mcapi\Entity\Transaction;
use Drupal\user\Entity\User;
use Drupal\mcapi\Mcapi;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Form builder to choose who referred you to the site.
 */
class ChooseReferrer extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'hamlets_referrer_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['referrer'] = [
      '#title' => $this->t('Who encouraged you to join this site?'),
      '#description' => $this->t('Tell us and they will receive a small reward.'),
      '#type' => 'entity_autocomplete',
      '#target_type' => 'user',
      '#selection_handler' => 'default:user',
      '#selection_settings' => [
        'include_anonymous' => 0,
        'filter' => [
          'type' => 'role',
          'role' => [RID_TRADER => RID_TRADER],
        ],
        'sort' => [
          'field' => 'name',
          'direction' => 'ASC',
        ],
      ],
      '#placeholder' => t('Name...'),
      '#tags' => FALSE,
    ];
    $form['new_user'] = [
      '#type' => 'value',
      '#value' => $form_state->getBuildInfo()['args'][0],
    ];
    $form['actions'] = [
      'submit' => [
        '#type' => 'submit',
        '#value' => $this->t('Pay'),
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);
    $uid = $form_state->getValue('referrer');
    $settings = \Drupal::config('cforge.settings');
    $props = [
      'payer' => $settings->get('referral_src'),
      'payee' => Mcapi::firstWalletIdOfEntity(User::load($uid)),
      'type' => 'auto',
      'worth' => $settings->get('referral_fee'),
      'description' => $this->t('Reward for referring new member @num', ['@num' => $form_state->getValue('new_user')]),
    ];
    $transaction = Transaction::create($props);
    if ($violations = $transaction->validate()) {
      $form_state->setErrorByName('referrer', t('Sorry, the payment failed.'));
      foreach ($violations as $violation) {
        \Drupal::logger('cforge')->error($violation->getMessageTemplate(), $violation->getParameters());
      }
      return;
    }
    $form_state->set('mcapi_transaction', $transaction);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $form_state->get('mcapi_transaction')->save();
    $uid = $form_state->getValue('new_user');
    \Drupal::keyValue('new_user_reference')->delete($uid);
    // @this should work but I'm  not sure if this is the best way to clear the block access cache
    Cache::invalidateTags(['user:' . $uid]);
    $payee_user = $form_state->get('mcapi_transaction')->payee->entity->getHolder();
    \Drupal::logger('cforge')->info(
      'Reward given to %name for referring %new_name',
      [
        '%name' => $payee_user->getDisplayName(),
        '%new_name' => User::load($form_state->getValue('new_user'))->getDisplayName(),
      ]
    );
    drupal_set_message(
      $this->t('%name has been given %amount for referring you!'),
      [
        '%name' => $payee_user->toLink(),
        '%amount' => $form_state->get('mcapi_transaction')->worth->toString(),
      ]
    );
  }

}
