<?php

namespace Drupal\cforge\Forms;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Send an email to all members of the site, regardless of preferences.
 */
class ContactAll extends FormBase {

  const SCOPE_ALL = 1;
  const SCOPE_ME = 0;

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'cforge_contact_all';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['subject'] = [
      '#title' => $this->t('Message subject'),
      '#type' => 'textfield',
      '#weight' => 0,
    ];
    $form['body'] = [
      '#title' => $this->t('Message body'),
      '#description' => $this->t('The following tokens are some of the tokens available: [user:account-name], [user:firstname], [user:one-time-login-url]'),
      '#type' => 'textarea',
      '#rows' => 10,
      '#weight' => 1,
    ];
    $form['scope'] = [
      '#type' => 'radios',
      '#options' => [
        SELF::SCOPE_ALL => $this->t('TEST send only to me'),
        SELF::SCOPE_ME => $this->t('Send to all active members'),
      ],
      '#default_value' => 1,
      '#weight' => 2,
    ];
    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Send'),
      '#weight' => 3,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    if ($form_state->getValues('scope') == SELF::SCOPE_ME) {
      $this->send1mail(\Drupal::currentuser()->getEmail(), $form_state->getValue('subject'), $form_state->getValue('body'));
    }
    else {
      $batch = ['title' => 'Sending...'];
      $uids = \Drupal::entityQuery('user')->condition('status', 1)->execute();
      foreach (array_chunk($uids, 20) as $chunk) {
        $batch['operations'][] = [
          [$this, 'sendBatchMails'],
          [$form_state->getValue('subject'), $form_state->getValue('body')],
        ];
      }
    }
    batch_set($batch);
  }

  /**
   * Batch callback.
   */
  public function sendBatchMails($mails, $subject, $body) {
    foreach ($mails as $email) {
      $this->send1mail($mail, $subject, $body);
    }
  }

  /**
   * Send an email.
   */
  public function send1mail($mail, $subject, $body) {
    \Drupal::service('plugin.manager.mail')->mail(
      'cforge',
      'all',
      $mail,
      \Drupal::currentuser()->getPreferredLangcode(),
      ['subject' => $subject, 'body' => $body]
    );
  }

}
