<?php

namespace Drupal\cforge\Forms;

use CommerceGuys\Addressing\AddressFormat\AddressField;
use Drupal\mcapi\Entity\Wallet;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Form builder for Cforge settings.
 */
class Settings extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'cforge_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildform($form, $form_state);
    $config = $this->config('cforge.settings');
    $form['mail'] = [
      '#title' => $this->t('Mail'),
      '#type' => 'details',
      'mail_footer' => [
        '#title' => $this->t('Footer'),
        '#description' => $this->t('Appended to all mails sent by the site to members. HTML is allowed.'),
        '#type' => 'textarea',
        '#rows' => 4,
        '#default_value' => $config->get('mail_footer'),
      ],
      // The broadcast settings would go here.
      '#weight' => 2,
    ];
    $form['robots_protect'] = [
      '#title' => t('Conceal from search engines'),
      '#description' => t('Rewrites the robots.txt file'),
      '#type' => 'checkbox',
      '#default_value' => $config->get('robots_protect', 0),
      '#weight' => 4,
    ];
    $form['referral'] = [
      '#title' => t('Referral'),
      '#description' => t('This user will receive a reward when a new account is created'),
      '#type' => 'fieldset',
      'referral_fee' => [
        '#title' => t('Referral fee'),
        '#type' => 'worths_form',
        '#default_value' => $config->get('referral_fee'),
        '#config' => TRUE,
      ],
      'referral_src' => [
        '#title' => t('Referral source wallet'),
        '#description' => t('Wallet from which referal payments are drawn.'),
        '#type' => 'wallet_entity_auto',
        // Autocomplete field takes the whole entity
        '#default_value' => Wallet::load($config->get('referral_src')),
        '#weight' => 3,
      ],
      '#weight' => 3,
    ];
    $terminology = [];
    foreach ($config->get('terminology') as $key => $value) {
      $terminology[] = "$key|$value";
    }

    $form['terminology'] = [
      '#title' => $this->t('Replace words'),
      '#description' => $this->t('This is applied to all interface strings after translation. Not config or content. Case sensitive.'),
      '#type' => 'textarea',
      '#default_value' => implode("\n", $terminology),
      '#element_validate' => [[get_class(), 'validateTerminology']],
      '#rows' => 8,
    ];
    return $form;
  }

  /**
   * Element validation callback.
   */
  public static function validateTerminology(&$element, $form_state) {
    try {
      $array = static::extractAllowedValues($element['#value']);
    }
    catch (\Exception $e){
      $form_state->setError($element, $e->getMessage());
    }
    $form_state->setValue('terminology', $array);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    // can't have a referral fee without a wallet.
    if (!empty_worths_config($form_state->getValue('referral_fee'))) {
      if (!$form_state->getValue('referral_src')) {
        $form_state->setError(
          $form['referral']['referral_src'],
          t('No wallet has been specified to pay the referral fee')
        );
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $values = $form_state->getValues();
    $config = $this->configFactory->getEditable('cforge.settings');
    $config
      ->set('mail_footer', $values['mail_footer'])
      ->set('robots_protect', $values['robots_protect'])
      ->set('referral_fee', $form_state->getValue('referral_fee'))
      ->set('referral_src', $form_state->getValue('referral_src'))
      ->set('terminology', $form_state->getValue('terminology'))
      ->save();
  }

  /**
   * {@inheritdoc}
   */
  public function getEditableConfigNames() {
    return ['cforge.settings'];
  }

  /**
   * Extracts the allowed values array from the terminology element.
   *
   * @param string $string
   *   The raw string to extract values from.
   *
   * @return array
   *   The array of extracted key/value pairs, or NULL if the string is invalid.
   *
   * @throws Exception
   */
  protected static function extractAllowedValues($string) {
    $values = [];
    $list = explode("\n", $string);
    $list = array_map('trim', $list);
    $list = array_filter($list, 'strlen');
    $generated_keys = $explicit_keys = FALSE;
    foreach ($list as $position => $text) {
      $matches = [];
      if (preg_match('/(.+)+\|(.+)/', $text, $matches)) {
        // Trim key and value to avoid unwanted spaces issues.
        $key = trim($matches[1]);
        $value = trim($matches[2]);
        $explicit_keys = TRUE;
      }
      else {
        throw new \Exception($this->t('Could not parse terminology field'));
      }
      $values[$key] = $value;
    }
    return $values;
  }

}
