<?php

namespace Drupal\cforge\Installer;

use Drupal\node\Entity\Node;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Choose between different Cforge flavours.
 */
class ChooseFlavour extends FormBase {

  private $installer;
  private $accountSwitcher;
  private $languageManager;

  /**
   * Constructor.
   */
  public function __construct($module_installer, $account_switcher, $language_manager) {
    $this->installer = $module_installer;
    $this->accountSwitcher = $account_switcher;
    $this->languageManager = $language_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('module_installer'),
      $container->get('account_switcher'),
      $container->get('language_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'cforge_flavour_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form_state->loadInclude('hamlets', 'module');
    $form['#title'] = t('Choose flavour');
    foreach (CFORGE_FLAVOURS as $mod => $name) {
      $writable[$mod] = $this->makeDefaultContentPath($mod);
    }
    $options = ['upgrade' => 'Hamlets upgrade from Drupal 7'] + CFORGE_FLAVOURS;
    asort($options);
    $form['flavour'] = [
      '#title' => $this->t('Flavour', [], ['context' => 'Installation variety']),
      '#type' => 'radios',
      '#options' => $options,
      '#default_value' => 'hamlets',
      '#required' => TRUE,
      '#weight' => 4,
    ];
    foreach (CFORGE_FLAVOURS as $mod => $name) {
      if (!$writable[$mod]) {
        $form['flavour']['#options'][$mod] .= '('.t('English only').')';
      }
    }

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => 'submit',
      '#weight' => 10,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $flavour_module = $form_state->getValue('flavour');
    if ($flavour_module == 'upgrade') {
      $this->installer->install([
        'migrate',
        'migrate_drupal',
        'migrate_drupal_ui',
        'migrate_plus',
        'migrate_tools',
        'migrate_upgrade',
        'options'
      ]);
      //avoid the setup and go to /upgrade
      $flavour_module = 'hamlets';
    }
    else {
      $this->installer->install(['hal', 'default_content']); //ready for enabling the flavour module
    }

    module_set_weight($flavour_module, 100);
    $this->configFactory()->getEditable('cforge.settings')
      ->set('flavour', $flavour_module)
      ->save();

    $this->makeDefaultContentPath($flavour_module);
    cforge_clear_preexisting_config($flavour_module);
    $this->installer->install([$flavour_module]);
    $this->installer->uninstall(['hal', 'default_content']);
    unlink($this->defaultContentPath($flavour_module));


    // Assume all the default content is public
    // set imported nodes to public using the cforge flag for that purpose.
    // set node owners to 1 because AccountSwitcher doesn't work.
    foreach (Node::loadMultiple() as $node) {
      $node->setOwnerId(1);
      if ($node->bundle() == 'page') {
        cforge_node_set_anonaccess($node->id(), TRUE);
      }
      // To get around a deficiency in default_content which fails to export Path
      // aliases we're going to set drupal's homepage using the hacked uuid.
      // @see https://www.drupal.org/node/2649646
      // @todo this is now solved
     if (substr($node->uuid(), 0, 8) == '00000000') {
        $config = $this->configFactory()->getEditable('system.site');
        $temp = $config->get('page');
        // The crux of the matter.
        $temp['front'] = '/node/' . $node->id();
        $config->set('page', $temp)->save();
      }
      $node->save();
    }
  }

  private function defaultContentPath($flavour_module) {
    return DRUPAL_ROOT . '/' . drupal_get_path('module', $flavour_module) . '/content';
  }

  private function makeDefaultContentPath($flavour_module) {
    $content_path = $this->defaultContentPath($flavour_module);
    @unlink($content_path);
    $curr_lang_id = $this->languageManager->getDefaultLanguage()->getId();
    $content_source = $content_path . '-' . $curr_lang_id;
    symlink($content_source, $content_path);
    if (file_exists($content_path)) {
      return TRUE;
    }
    drupal_set_message(t(
      "Unable to install default_content for @flavour because @dir is not writable",
      ['@dir' => $content_path, '@flavour' => $flavour_module]
    ));

  }

}
