<?php

namespace Drupal\cforge\Installer;

use Drupal\mcapi\Entity\Currency;
use Drupal\user\Entity\User;
use Drupal\taxonomy\Entity\Term;
use Drupal\taxonomy\Entity\Vocabulary;
use Drupal\Core\Render\Element;
use Drupal\Core\Url;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

ini_set('max_execution_time', 150);

/**
 * Configuration options for each flavour.
 */
class Setup extends FormBase {

  private $installer;
  private $accountSwitcher;
  private $countryRepository;
  private $entityTypeManager;
  protected $linkGenerator;

  /**
   * Constructor.
   */
  public function __construct($module_installer, $account_switcher, $link_generator, $country_repo, $entity_type_manager) {
    $this->installer = $module_installer;
    $this->accountSwitcher = $account_switcher;
    $this->linkGenerator = $link_generator;
    $this->countryRepository = $country_repo;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('module_installer'),
      $container->get('account_switcher'),
      $container->get('link_generator'),
      $container->get('address.country_repository'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'cforge_setup_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {

    $flavour = $this->config('cforge.settings')->get('flavour');
    if (!$flavour) {
      throw new \Exception('Cannot run set up without a flavour.');
    }
    $form_state->loadInclude($flavour, 'install');
    $form_state->loadInclude('cforge', 'install');
    $form['#title'] = t('Setup @flavour', ['@flavour' => $flavour]);

    $form['data_content'] = [
      '#title' => t('Prepare the site with'),
      '#type' => 'radios',
      '#options' => [
        'none' => t('No content'),
        'import' => t('CSV Import Tools'),
        'example' => t('Example data'),
      ],
      '#weight' => 0,
      'example' => [
        '#disabled' => !system_rebuild_module_data()['devel_generate'],
      ],
      '#required' => TRUE,
    ];
    if (!$this->config('system.date')->get('country.default')) {
      $form['site_default_country'] = [
        '#title' => $this->t('Country'),
        '#type' => 'select',
        '#options' => $this->countryRepository->getList(),
        '#required' => TRUE,
        '#weight' => 3,
      ];
    }

    $demo_data_class = cforge_demo_data_class_name();
    $form['neighbourhoods'] = [
      '#title' => $this->t('Neighbourhoods'),
      '#description' => $this->t('Limited list of possible neighbourhoods, one per line, in order. Leave blank if you plan to import data'),
      '#type' => 'textarea',
      '#default_value' => implode("\r\n", $demo_data_class::$hoods),
      '#weight' => 8,
      '#required' => 1,
      '#states' => [
        'visible' => [
          ':input[name=data_content]' => ['value' => 'none'],
        ],
      ],
    ];

    $vocab_names = array_keys($demo_data_class::vocabs());
    $form['default_categories'] = array(
      '#title' => $this->t('Preload ad categories'),
      '#description' => $this->t('Can be modified later'),
      '#type' => 'radios',
      '#options' => [0 => $this->t('- None -')] + array_combine($vocab_names, $vocab_names),
      '#default_value' => key($vocab_names),
      '#weight' => 9,
      '#states' => [
        'visible' => [
          ':input[name=data_content]' => ['value' => 'none'],
        ],
      ],
    );

    $form['currency_type'] = [
      '#title' => $this->t('Main currency type'),
      '#type' => 'radios',
      '#options' => [
        'hours' => $this->t('Hours'),
        'other' => $this->t('Other'),
      ],
      '#weight' => 6,
    ];
    $form['currency_name'] = [
      '#title' => $this->t('Currency name'),
      '#description' => $this->t('Use the plural'),
      '#type' => 'textfield',
      '#placeholder' => $this->t('Units'),
      '#weight' => 8,
      '#states' => [
        'visible' => [
          ':input[name="currency_type"]' => ['value' => 'other'],
        ],
      ],
    ];

    $form['actions'] = [
      'submit' => [
        '#type' => 'submit',
        '#value' => 'submit',
      ],
      '#weight' => 20,
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {

    static::setCountrySubmit($form, $form_state);

    $modules = $hoods = [];
    $demo_data_class = cforge_demo_data_class_name();
    $this->accountSwitcher->switchTo(User::load(1));

    if ($form_state->getValue('default_categories')) {
      $terms = $this->entityTypeManager
        ->getStorage('taxonomy_term')
        ->loadByProperties(['vid' => SMALLAD_CATEGORIES]);
      // Delete any existing terms in the vocab (one is installed by default)
      foreach ($terms as $term) {
        $term->delete();
      }
      $vocab = Vocabulary::load(SMALLAD_CATEGORIES);
      // Now import the terms into the vocab.
      $this->vocabImportRecursive(
        $vocab,
        $demo_data_class::vocabs()[$form_state->getValue('default_categories')]
      );
      // Thie hierarchy property.
      $vocab->save();
    }

    // Set up the currency by deleting the others.
    if ($c = Currency::load('veur')) {
      $c->delete();
    }
    if ($form_state->getValue('currency_type') == 'hours') {
      if ($c = Currency::load('cc')) {
        $c->delete();
      }
    }
    if ($form_state->getValue('currency_type') == 'cc') {
      if ($c = Currency::load('hhrs')) {
        $c->delete();
      }
    }
    if ($currs = Currency::loadMultiple()) {
      $curr = reset($currs);
      $curr->name = $form_state->getValue('currency_name') ?: 'Units';
      $symbol = strtoupper(substr($curr->name, 0, 1));
      $curr->format = ["<strong>$symbol</strong>", '9999'];
      $curr->save();
    }

    switch ($form_state->getValue('data_content')) {
      case 'import':
        $modules = ['cforge_import'];
        $hoods = array_unique(explode("\n", trim($form_state->getValue('neighbourhoods'))));
        break;

      case 'example':
        // Enable example modules and settings
        $modules = ['views_ui',
          'field_ui',
          'contextual',
          'config',
          'admin_toolbar_tools',
          'devel',
          'devel_generate',
        ];
        $this->configFactory()->getEditable('views.settings')
          ->set('ui.show.sql_query.enabled', 1)
          ->set('ui.always_live_preview', 1)
          ->set('ui.show.advanced_column', 1)
          ->set('ui.show.master_display', 1)
          ->save();
        $this->configFactory()->getEditable('cforge.settings')
          ->set('demo_mode', TRUE)
          ->save();
        $hoods = $demo_data_class::$hoods;
    }

    if ($hoods) {
      $demo_data_class::createNeighbourhoods($hoods);
    }
    $this->accountSwitcher->switchBack();

    if ($modules) {
      drupal_set_message('Installing modules: '.implode(', ', $modules), 'warning');
      $this->installer->install($modules);
    }

    cforge_backup();

    $form_state->setRedirectUrl(
      Url::fromUserInput($this->config('cforge.settings')->get('member_frontpage'))
    );

    if ($form_state->getValue('data_content') == 'example') {
      $batch = [
        'title'  => t('Setting up Hamlets'),
        'operations' => []
      ];
      $batch = cforge_generate_demodata_batch();
      $batch['finished'] = 'cforge_finish_setup';
      batch_set($batch);
    }
    else {
      cforge_finish_setup();
    }

  }


  /**
   * Import the terms in a vocabulary or a level of a vocab.
   */
  public function vocabImportRecursive(Vocabulary $vocab, array $termArray, $parent = 0) {
    $w = $hierarchy = 0;
    foreach (Element::Children($termArray) as $name) {
      $props = [
        'name' => $name,
        'weight' => $w++,
        'vid' => $vocab->id(),
        'parent' => $parent,
        'format' => 'plain_text',
      ];
      $term = Term::Create($props);
      $term->save();
      if ($termArray['#children']) {
        // @todo switch over these lines on 8.2.0
        //$vocab->setHierarchy(VocabularyInterface::HIERARCHY_SINGLE);
        $vocab->setHierarchy(TAXONOMY_HIERARCHY_SINGLE);
        $this->vocabImportRecursive($vocab, $termArray[$name], $term->tid);
      }
    }
  }

  /**
   * Form submit callback
   */
  public static function setCountrySubmit($form, $form_state) {
    if ($country_code = $form_state->getValue('site_default_country')) {
      module_load_install('cforge');
      cforge_set_country($country_code);
    }
  }

}
