<?php

namespace Drupal\cforge\Plugin\Block;

use Drupal\user\Entity\User;
use Drupal\user\Entity\Role;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Block\BlockBase;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Shows usernames and passwords for first account of each role
 *
 * @Block(
 *   id = "demo_logins",
 *   admin_label = @Translation("Demo login details"),
 *   category = @Translation("Hamlets")
 * )
 */
class DemoLogins extends BlockBase implements ContainerFactoryPluginInterface {

  private $cforgeSettings;

  private $entityQuery;

  /**
   * {@inheritdoc}
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, $config_factory, $entity_query) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->cforgeSettings = $config_factory->get('cforge.settings');
    $this->entityQuery = $entity_query;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory'),
      $container->get('entity.query')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function label() {
    return $this->t('Demo logins');
  }

  /**
   * {@inheritdoc}
   *
   * @todo Only access the block when we are in demo mode
   */
  protected function blockAccess(AccountInterface $account) {
    $result = ($this->cforgeSettings->get('demo_mode') && $account->isAnonymous())
      ? AccessResult::allowed()
      : AccessResult::forbidden();
    return $result->addCacheableDependency($this->cforgeSettings);
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    return [[
      '#theme' => 'item_list',
      '#title'  => $this->t("Login with any of following usernames and password 'demo'"),
      '#items' => $this->getNames(),
      // Need some css to remove the bullet points
      '#type' => 'ul',
    ]];
  }

  public function getNames() {
    // Get a trader who isn't a committee member.
    $uids = $this->entityQuery->get('user')
      ->condition('uid', 1, '<>')
      ->condition('status', 1)
      ->condition('roles', 'trader')
      ->condition('roles', 'committee', '<>')
      ->execute();
    shuffle($uids);
    $names['trader'] = reset($uids);

    $uids = $this->entityQuery->get('user')
      ->condition('uid', 1, '<>')
      ->condition('status', 1)
      ->condition('roles', 'committee')
      ->execute();
    shuffle($uids);
    $names['committee'] = reset($uids);

    foreach ($names as $rid => $uid) {
      $names[$rid] = User::load($uid)->getUserName() . ' ('.Role::load($rid)->label().')';
    }
    return $names;
  }

}
